'Race example V1.0
'By David Powell
'http://www.loadcode.co.uk
'Released into the public domain

'Car graphic from:
'http://openclipart.org/detail/2056/pigeau-by-glibersat

'Tree graphic from:
'http://openclipart.org/detail/184546/tree-04-by-jpenrici-184546

'This example uses mask rotation on a single object and mask scaling on multiple objects,
'along with mask to mask collision detection.
'The car mask is rotated on each update to match the angle the car is currently at.
'The trees are created at different sizes. When a tree is created the tree mask is scaled
'to the same size and cached with the rest of the tree data.
'This means that, although there are lots of objects on-screen at the same time, a maximum
'of one mask rotation and two mask scales are performed each update. This keeps processing
'time to a minimum.
'As collision detection is fast, the car can be collision checked against every tree
'on each update.

'#FAST_SYNC_PROJECT_DATA = True

Strict

Import mojo
Import brl.pool
Import collisionmask

Public
	'Summary: Start the RaceApp application
	Function Main:Int()
		New RaceApp()
		Return 0
	End
	
Public
	'Summary: The Race application
	Class RaceApp Extends App
		Private
			'The car data
			Field carImage:Image 'The car image
			Field carMasterMask:MasterMask 'The car master mask to use as a rotation source
			Field carRotatedMask:Mask 'The car mask rotated to the current car angle
			
			'The tree data
			Field treeImage:Image 'The tree image
			Field treeMasterMask:MasterMask 'The tree master mask used as a scaling source
			Field checkTreeMask:Mask 'A tree mask to use to check for new tree overlaps
			
			'The car state
			Field carX:Int 'The X position of the car is set to the centre of the screen
			Const CarY:Int = 70 'The Y position of the car is fixed
			Field carAngle:Float = 0 'The current angle of the car
			Const CarSpeed:Int = 4 'The speed (in pixels) that the car moves each update
			
			'The current tree information
			Const TreeSpawnChance:Float = 0.1 '10% chance of spawning a new tree every update
			Const MaxTrees:Int = 20 'The maximum number of trees on-screen at once
			Field treeList:List<Tree> 'The list of trees currently on-screen
			
			'The crash information
			Const CrashWait:Int = 60 * 2 'How long to pause the game after a crash (2 seconds)
			Field carCrashWait:Int = CrashWait 'The amount of time remaining after a crash
			Field crashed:Bool = False 'The crashed flag
		
		Public
			'Summary: When the application is created; initialise everything
			Method OnCreate:Int()
				'As the images are loaded with the Image.XYPadding flag set, when the
				'masks were generated they had a border of 1 pixel trimmed so that the
				'size is the same as the image after padding removal.
				'If a sprite atlas was used then mask trimming would not be required.
			
				carImage = LoadImage("monkey://data/car.png", 1, Image.XYPadding | Image.MidHandle)
				If carImage = Null Then Error("Cannot load car.png")
				
				carMasterMask = LoadMask("monkey://data/car.cmsk", Mask.MidHandle)
				If carMasterMask = Null Then Error("Cannot load car.cmsk")
				
				treeImage = LoadImage("monkey://data/tree.png", 1, Image.XYPadding | Image.MidHandle)
				If treeImage = Null Then Error("Cannot load tree.png")
				
				treeMasterMask = LoadMask("monkey://data/tree.cmsk", Mask.MidHandle)
				If treeMasterMask = Null Then Error("Cannot load tree.cmsk")
				
				'The tree pool is now initialised and the on-screen tree list is created
				'The tree master mask is passed to the Tree object so that new trees
				'can create a scaled mask of the correct size based on the original master mask.
				Tree.InitPool(treeMasterMask, MaxTrees)
				treeList = New List<Tree>
				
				'The car rotated mask and check tree mask are pre-allocated so that
				'New is not called in the main game loop
				carRotatedMask = carMasterMask.PreAllocateMask(1.0, 1.0)
				checkTreeMask = treeMasterMask.PreAllocateMask(1.0, 1.0)
				
				Return 0
			End
			
		Public
			'Summary: Update the current state of the application
			Method OnUpdate:Int()
				'If the car has crashed, wait and then reset the game
				If crashed = True
					'Reduce the car crash wait delay
					carCrashWait -= 1
					
					'If the wait delay has ended; reset the game
					If carCrashWait <= 0
						crashed = False
						carCrashWait = CrashWait
						
						'Destroy all the tree objects
						For Local tree:Tree = EachIn treeList
							tree.Destroy()
						Next
						
						'Clear the list of on-screen trees
						treeList.Clear()
					EndIf
					
					Return 0
				EndIf
			
				'Set the car X position to be the centre of the display
				carX = DeviceWidth() / 2
			
				'Create a new tree
				If treeList.Count() < MaxTrees
					If Rnd() < TreeSpawnChance
						Local xPos:Int = Rnd(-DeviceWidth(), DeviceWidth() * 2)
						Local yPos:Int = DeviceHeight() +treeImage.Height() / 2
						Local size:Float = Rnd(0.5, 1.0)
						
						'Scale the tree master mask to match the size of the proposed new tree
						treeMasterMask.Scale(size, size, checkTreeMask)
						
						'Default the overlap flag to false
						Local overlap:Bool = False
						
						'Check that the proposed new tree does not overlap any of the existing trees
						For Local checkTree:Tree = EachIn treeList
							If checkTreeMask.CollideMask(xPos, yPos, checkTree.mask, checkTree.x, checkTree.y) = True
								overlap = True
								Exit
							EndIf
						Next

						'If there is no overlap then the new tree can be created
						If overlap = False
							Local tree:Tree = Tree.Create(xPos, yPos, size)
							treeList.AddLast(tree)
						EndIf
					EndIf
				EndIf
				
				'Set the angle of the car to point towards the mouse pointer
				carAngle = ATan2(MouseX() -carX, MouseY() -CarY)
				
				'Clamp the car angle to between -45 and 45 degrees, so that the car
				'cannot be driven sideways or backwards
				carAngle = Clamp(carAngle, -45.0, 45.0)
				
				'Rotate the car master mask to reflect the current car angle
				carMasterMask.ScaleAndRotate(1.0, 1.0, carAngle, carRotatedMask)
				
				'Update the position of each tree and check for collisions with the car
				For Local tree:Tree = EachIn treeList
					'The tree vertical movement depends on the car speed.
					tree.y -= CarSpeed
					
					'The tree horizontal movement depends on the current angle of the car
					tree.x += -Int(carAngle) / 10
					
					'If the tree is off the top of the screen then remove it
					If tree.y < - treeImage.Height() / 2
						treeList.Remove(tree)
						tree.Destroy()
					EndIf
					
					'Check to see if this tree has collided with the car
					If carRotatedMask.CollideMask(carX, CarY, tree.mask, tree.x, tree.y) = True
						crashed = True
					EndIf
				Next
				
				Return 0
			End
			
		Public
			'Summary: Render the current state of the application
			Method OnRender:Int()
				'Choose the background colour depending on the crashed flag
				If crashed = True
					Cls(69, 34, 9)
				Else
					Cls(139, 69, 19)
				EndIf
				
				'Draw each tree on-screen
				For Local tree:Tree = EachIn treeList
					DrawImage(treeImage, tree.x, tree.y, 0.0, tree.size, tree.size)
					'tree.mask.DebugDraw(tree.x, tree.y, 0.5)
				Next
				
				'Draw the car
				DrawImage(carImage, carX, CarY, carAngle, 1.0, 1.0)
				'carRotatedMask.DebugDraw(carX, CarY, 0.5)
				
				Return 0
			End
	
	End
	
Public
	'Summary: The Tree object
	Class Tree
		Public
			'The attributes of an Tree
			Field x:Int
			Field y:Int
			Field size:Float
			Field mask:Mask
			
		Private
			'Global information about all Trees
			'A tree pool is used so that New is not called in the main game loop
			Global treePool:Pool<Tree>
			Global treeMasterMask:MasterMask
			Global maxSize:Float = 1.0
			
		Public
			'Summary: Initialise the tree pool with Tree objects
			Function InitPool:Void(masterMask:MasterMask, maxTrees:Int)
				'Store the tree master mask for when New() is called
				treeMasterMask = masterMask
				treePool = New Pool<Tree>(maxTrees)
			End
			
		Public
			'Sumary: Create a Tree
			Function Create:Tree(x:Int, y:Int, size:Float)
				'Tree objects are retrieved from the tree pool
		        Return treePool.Allocate().Init(x, y, size)
		    End
			
		Public
			'Summary: Destroy the Tree
			Method Destroy:Void()
				'Tree objects are returned to the tree pool
		        treePool.Free(Self)
		    End
			
		Public
			'Summary: Initialse a Tree
			Method New()
				'A mask for this tree is pre-allocated from the master mask
				mask = treeMasterMask.PreAllocateMask(1.0, 1.0)
			End
			
		Private
			'Summary: Initialise the Tree
			Method Init:Tree(x:Int, y:Int, size:Float)
				Self.x = x
				Self.y = y
				Self.size = size
				treeMasterMask.Scale(size, size, Self.mask)
				Return Self
			End
	End

